
import os
import logging
from typing import Optional

from fastapi import FastAPI, HTTPException
from fastapi.middleware.cors import CORSMiddleware

from app.login import login_router
from app.proxy_router import router as proxy_router
from app.media import media_router
from app.post import post_router
from app.createa import router as create_account_router
from app.check import router as check_router
from app.task_manager import background_tasks

# Setup logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

app = FastAPI(title="Instagram Login API")

# Add CORS middleware
app.add_middleware(
    CORSMiddleware,
    allow_origins=["http://localhost:5173", "http://52.205.253.74:5173", "http://172.31.87.147:5173"],  # Frontend URLs
    allow_credentials=True,
    allow_methods=["*"],  # Allow all methods
    allow_headers=["*"],  # Allow all headers
)

# Include routers
app.include_router(create_account_router)
app.include_router(check_router)
app.include_router(login_router, prefix="/login", tags=["Login"])
app.include_router(media_router, prefix="/media", tags=["Media"])
app.include_router(post_router, prefix="/post", tags=["Post"])
# Include router from createa.py for account creation
app.include_router(proxy_router)
# Include router from check.py for username checking


@app.get("/")
async def root():
    return {
        "name": "Instagram Login API",
        "description": "API for Instagram account login operations",
        "routes": [
            "POST /login/automatic",
            "POST /login/from-text",
            "POST /login/from-file",
            "POST /login/cancel/{request_id}",
            "GET /jobs",
            "GET /jobs/{job_id}",
            "POST /media/carousel",
            "GET /media/sessions",
            "POST /post/upload",
            "GET /post/sessions"
        ],
        "sessions_dir": os.path.abspath("sessions")
    }


@app.get("/jobs/{job_id}")
async def get_job_status(job_id: str):
    job = background_tasks.get_task(job_id)
    if not job:
        raise HTTPException(status_code=404, detail="Job not found")
    return job


@app.get("/jobs")
async def list_jobs(name: Optional[str] = None):
    return background_tasks.list_tasks(name)

if __name__ == "__main__":
    import uvicorn
    
    print("ðŸš€ Starting Instagram API Server...")
    print("   Server will be available at: http://127.0.0.1:8000")
    print("   API docs at: http://127.0.0.1:8000/docs")
    print("   Press Ctrl+C to stop the server")
    print()
    
    uvicorn.run(
        "main:app", 
        host="127.0.0.1", 
        port=7000, 
        reload=True,
        reload_dirs=["app", "."],
        log_level="info"
    )
