import asyncio
import traceback
import uuid
from datetime import datetime, timezone
from typing import Any, Awaitable, Callable, Dict, List, Optional


class BackgroundTaskManager:
    """
    Simple in-memory background task manager that tracks async tasks
    and exposes their status/results through an API.
    """

    def __init__(self) -> None:
        self._tasks: Dict[str, Dict[str, Any]] = {}
        self._lock = asyncio.Lock()

    async def create_task(
        self,
        name: str,
        coro_func: Callable[..., Awaitable[Any]],
        *args,
        job_id: Optional[str] = None,
        **kwargs,
    ) -> Dict[str, Any]:
        """
        Schedule a coroutine to run in the background.

        Args:
            name: Human readable name for the job (e.g., "media:carousel").
            coro_func: Async callable to run.
            *args/**kwargs: Arguments passed to the callable.
            job_id: Optional custom job ID (used when we already have a request_id).

        Returns:
            Dict representing the scheduled job (id, status, etc.).
        """
        if job_id is None:
            job_id = str(uuid.uuid4())
        else:
            # Avoid accidental overwrites if the caller reuses IDs
            async with self._lock:
                if job_id in self._tasks:
                    raise ValueError(f"Job with id {job_id} already exists")

        job_info: Dict[str, Any] = {
            "id": job_id,
            "name": name,
            "status": "queued",
            "result": None,
            "error": None,
            "created_at": datetime.now(timezone.utc).isoformat(),
            "started_at": None,
            "completed_at": None,
        }

        async with self._lock:
            self._tasks[job_id] = job_info

        async def runner():
            try:
                job_info["status"] = "running"
                job_info["started_at"] = datetime.now(timezone.utc).isoformat()
                result = await coro_func(*args, **kwargs)
                job_info["result"] = result
                job_info["status"] = "completed"
            except Exception as exc:  # pragma: no cover - logging helper
                job_info["error"] = str(exc)
                job_info["traceback"] = traceback.format_exc()
                job_info["status"] = "failed"
            finally:
                job_info["completed_at"] = datetime.now(timezone.utc).isoformat()

        asyncio.create_task(runner())
        return job_info

    def get_task(self, job_id: str) -> Optional[Dict[str, Any]]:
        return self._tasks.get(job_id)

    def list_tasks(self, name: Optional[str] = None) -> List[Dict[str, Any]]:
        if name:
            return [task for task in self._tasks.values() if task.get("name") == name]
        return list(self._tasks.values())


background_tasks = BackgroundTaskManager()

