"""
FastAPI Router for Proxy Management
API endpoints for managing proxies in proxies.txt file
"""
from fastapi import APIRouter, HTTPException
from pydantic import BaseModel, Field
from app.utils.proxy_manager import (
    read_proxies,
    add_proxy,
    remove_proxy,
    get_random_proxy
)


# Create router for proxy management
router = APIRouter(prefix="/api/proxies", tags=["Proxy Management"])


# Request/Response models
class ProxyRequest(BaseModel):
    """Request model for proxy operations."""
    proxy: str = Field(..., description="Proxy string (e.g., username:password@host:port)")


@router.get("/")
async def get_proxies():
    """
    Get all proxies from proxies.txt file.
    
    Returns:
        List of all proxies
    """
    try:
        proxies = read_proxies()
        return {
            "message": "Proxies retrieved successfully",
            "proxies": proxies,
            "count": len(proxies),
            "success": True
        }
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error reading proxies: {str(e)}")


@router.get("/random")
async def get_random():
    """
    Get a random proxy from proxies.txt file.
    
    Returns:
        Random proxy string
    """
    try:
        proxy = get_random_proxy()
        if proxy is None:
            raise HTTPException(status_code=404, detail="No proxies available")
        return {
            "message": "Random proxy retrieved successfully",
            "proxy": proxy,
            "success": True
        }
    except HTTPException:
        raise
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error getting random proxy: {str(e)}")


@router.post("/add")
async def add_proxy_endpoint(request: ProxyRequest):
    """
    Add a proxy to proxies.txt file.
    
    Args:
        request: Proxy request with proxy string
        
    Returns:
        Response with success status
    """
    try:
        success = add_proxy(request.proxy)
        if not success:
            raise HTTPException(status_code=400, detail="Failed to add proxy (may already exist)")
        return {
            "message": "Proxy added successfully",
            "proxy": request.proxy,
            "success": True
        }
    except HTTPException:
        raise
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error adding proxy: {str(e)}")


@router.delete("/remove")
async def remove_proxy_endpoint(request: ProxyRequest):
    """
    Remove a proxy from proxies.txt file.
    
    Args:
        request: Proxy request with proxy string to remove
        
    Returns:
        Response with success status
    """
    try:
        success = remove_proxy(request.proxy)
        if not success:
            raise HTTPException(status_code=404, detail="Proxy not found")
        return {
            "message": "Proxy removed successfully",
            "proxy": request.proxy,
            "success": True
        }
    except HTTPException:
        raise
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error removing proxy: {str(e)}")

