import os
import json
import glob
import csv
import logging
import asyncio
import random
import tempfile
import uuid
import shutil
from pathlib import Path
from typing import Dict, List, Optional, Tuple, Union
from datetime import datetime

import httpx
from fastapi import APIRouter, HTTPException, Body
from pydantic import BaseModel
from app.task_manager import background_tasks

# Setup logging first (needed for patch logging)
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger("media_app")
logger.setLevel(logging.INFO)
if not logger.handlers:
    handler = logging.StreamHandler()
    handler.setLevel(logging.INFO)
    formatter = logging.Formatter("%(asctime)s - %(levelname)s - %(message)s")
    handler.setFormatter(formatter)
    logger.addHandler(handler)

try:
    from instagrapi import Client
    from instagrapi.exceptions import (
        ClientLoginRequired,
        PleaseWaitFewMinutes,
        ChallengeRequired,
        RateLimitError,
    )
except ImportError:
    raise ImportError("instagrapi is required. Install it with: pip install instagrapi")

# Get paths (same as login.py)
SESSIONS_DIR = os.path.join(os.path.dirname(os.path.dirname(__file__)), "sessions")
VERIFY_CSV = os.path.join(os.path.dirname(os.path.dirname(__file__)), "verify.csv")
PROXIES_TXT = os.path.join(os.path.dirname(os.path.dirname(__file__)), "proxies.txt")
NOACTIVE_CSV = os.path.join(os.path.dirname(os.path.dirname(__file__)), "noactive account.csv")

# Cache for loaded proxies
_proxies_cache: Optional[List[str]] = None

# Create router
media_router = APIRouter()


class SimpleCarouselRequest(BaseModel):
    sessions: str  # Comma-separated: "session1,session2,session3"
    urls: str  # Comma-separated: "url1,url2,url3"
    caption: Optional[str] = ""  # Single caption for all posts
    captions: Optional[str] = None  # Alias for caption (for backward compatibility)
    order_name: Optional[str] = ""
    
    def get_caption(self) -> str:
        """Get caption from either 'caption' or 'captions' field (captions takes precedence if both provided)"""
        if self.captions:
            return self.captions.strip() if self.captions.strip() else ""
        return self.caption.strip() if self.caption and self.caption.strip() else ""


class CarouselPostResponse(BaseModel):
    id: str
    status: str
    created_at: str
    updated_at: str
    completed_at: Optional[str]
    results: List[Dict]
    summary: Dict


class BackgroundJobResponse(BaseModel):
    job_id: str
    status: str
    message: str
    total_sessions: int
    total_media: int


def mark_account_as_suspended(username: str, reason: str = "Session expired - relogin failed") -> None:
    """
    Mark account as suspended by removing from verify.csv, deleting session file, and adding to noactive account.csv
    """
    try:
        # Remove from verify.csv if it exists
        if os.path.exists(VERIFY_CSV):
            temp_path = f"{VERIFY_CSV}.tmp"
            removed = False
            with open(VERIFY_CSV, 'r', encoding='utf-8') as src, open(temp_path, 'w', newline='', encoding='utf-8') as dst:
                reader = csv.DictReader(src)
                fieldnames = reader.fieldnames or ["username", "password", "email", "email_password"]
                writer = csv.DictWriter(dst, fieldnames=fieldnames)
                writer.writeheader()
                for row in reader:
                    if row.get('username', '').strip() != username:
                        writer.writerow(row)
                    else:
                        removed = True
                        # Get account details before removing
                        account_data = {
                            'username': row.get('username', '').strip(),
                            'password': row.get('password', '').strip(),
                            'email': row.get('email', '').strip(),
                            'email_password': row.get('email_password', '').strip()
                        }
            
            if removed:
                os.replace(temp_path, VERIFY_CSV)
                logger.warning(f"🚫 Removed {username} from verify.csv (marked as suspended)")
                
                # Delete session file if it exists
                session_file = os.path.join(SESSIONS_DIR, f"{username}_session.json")
                if os.path.exists(session_file):
                    try:
                        os.remove(session_file)
                        logger.warning(f"🗑️ Deleted session file for {username}: {session_file}")
                    except Exception as session_error:
                        logger.error(f"Failed to delete session file for {username}: {session_error}")
                
                # Add to noactive account.csv
                if os.path.exists(NOACTIVE_CSV):
                    file_exists = os.path.getsize(NOACTIVE_CSV) > 0
                else:
                    file_exists = False
                
                with open(NOACTIVE_CSV, "a", newline="", encoding="utf-8") as f:
                    writer = csv.DictWriter(f, fieldnames=["username", "password", "authfa"])
                    if not file_exists:
                        writer.writeheader()
                    writer.writerow({
                        "username": account_data.get('username', ''),
                        "password": account_data.get('password', ''),
                        "authfa": ""  # No 2FA secret
                    })
                logger.warning(f"🚫 Added {username} to noactive account.csv - Reason: {reason}")
            else:
                # File was created but account not found, remove temp file
                if os.path.exists(temp_path):
                    os.remove(temp_path)
        else:
            logger.warning(f"verify.csv not found, cannot remove {username}")
    except Exception as e:
        logger.error(f"Error marking account {username} as suspended: {e}")


def get_account_from_verify_csv(username: str) -> Optional[Dict]:
    """
    Get account details from verify.csv by username.
    Returns dict with username, password, email, email_password
    """
    if not os.path.exists(VERIFY_CSV):
        logger.warning(f"verify.csv not found: {VERIFY_CSV}")
        return None
    
    try:
        with open(VERIFY_CSV, 'r', encoding='utf-8') as f:
            reader = csv.DictReader(f)
            for row in reader:
                if row.get('username', '').strip() == username:
                    return {
                        'username': row.get('username', '').strip(),
                        'password': row.get('password', '').strip(),
                        'email': row.get('email', '').strip(),
                        'email_password': row.get('email_password', '').strip()
                    }
        logger.warning(f"Username {username} not found in verify.csv")
        return None
    except Exception as e:
        logger.error(f"Error reading verify.csv: {e}")
        return None


async def relogin_account_with_instagrapi(username: str) -> bool:
    """
    Relogin account using instagrapi (which handles challenges better).
    Uses credentials from verify.csv automatically - NO PROMPTS.
    Returns True if successful, False otherwise.
    """
    try:
        account = get_account_from_verify_csv(username)
        if not account:
            logger.error(f"Cannot relogin {username}: account not found in verify.csv")
            return False
        
        logger.info(f"🔄 Attempting to relogin {username} using instagrapi...")
        
        # Import instagrapi Client and dependencies
        from instagrapi import Client as InstagrapiClient
        from instagrapi.mixins.challenge import ChallengeChoice
        from app.login import (
            get_code_from_email_for_challenge, 
            get_random_proxy as get_proxy_for_login, 
            DEFAULT_IMAP_PASSWORD,
            generate_random_user_agent,
            ensure_sessions_dir
        )
        
        # Create instagrapi client
        cl = InstagrapiClient()
        
        # Configure client settings (same as in login.py)
        cl.set_settings({
            "app_version": "269.0.0.18.75",
            "android_version": 26,
            "android_release": "8.0.0",
            "dpi": "480dpi",
            "resolution": "1080x1920",
            "manufacturer": "Xiaomi",
            "device": "capricorn",
            "model": "MI 5s",
            "cpu": "hexa-core",
            "version_code": "301"
        })
        
        # Set user agent
        cl.set_user_agent(generate_random_user_agent())
        
        # Get random proxy and ensure it has http:// scheme
        proxy_url = get_proxy_for_login()
        if proxy_url:
            # Ensure proxy has http:// scheme (instagrapi requires it)
            if not proxy_url.startswith(('http://', 'https://', 'socks4://', 'socks5://')):
                proxy_url = f"http://{proxy_url}"
            try:
                cl.set_proxy(proxy_url)
                logger.info(f"🔥 Using proxy for relogin: {proxy_url[:50]}...")
            except Exception as e:
                logger.warning(f"Failed to set proxy for relogin: {e}")
                logger.warning(f"Proxy format was: {proxy_url[:50]}...")
        
        # Set up challenge handler (MUST be set before login to avoid prompts)
        email_address = account.get('email', '')
        email_password = account.get('email_password', DEFAULT_IMAP_PASSWORD)
        insta_password = account.get('password', '')
        
        # Set password handler to automatically return password from verify.csv - NO PROMPTS
        def password_handler(insta_username):
            """Automatically return password from verify.csv - NO PROMPTS"""
            logger.info(f"🔐 Password requested for {insta_username}, returning from verify.csv")
            return insta_password
        
        # Monkey-patch both input() and getpass.getpass() to prevent password prompts
        import builtins
        import getpass
        
        original_input = builtins.input
        original_getpass = getpass.getpass
        
        def auto_input(prompt=""):
            prompt_lower = prompt.lower()
            if "password" in prompt_lower and (username in prompt_lower or username.lower() in prompt_lower):
                logger.info(f"🔐 Intercepted input() password prompt for {username}, returning from verify.csv")
                return insta_password
            # For other prompts, use original input (shouldn't happen)
            return original_input(prompt)
        
        def auto_getpass(prompt="", stream=None):
            prompt_lower = prompt.lower()
            if "password" in prompt_lower and (username in prompt_lower or username.lower() in prompt_lower):
                logger.info(f"🔐 Intercepted getpass() password prompt for {username}, returning from verify.csv")
                return insta_password
            # For other prompts, use original getpass (shouldn't happen)
            return original_getpass(prompt, stream)
        
        # Apply monkey-patches
        builtins.input = auto_input
        getpass.getpass = auto_getpass
        logger.info(f"✅ Monkey-patched input() and getpass() for {username} - NO PROMPTS")
        
        # Store references to restore later
        cl._original_input = original_input
        cl._original_getpass = original_getpass
        
        # Also try to set password handler (if instagrapi supports it)
        if hasattr(cl, 'password_handler'):
            cl.password_handler = password_handler
            logger.info(f"✅ Password handler also set for {username}")
        
        if email_address:
            def challenge_code_handler(insta_username, choice):
                """Handle Instagram challenge codes via IMAP - NO PROMPTS"""
                if choice == ChallengeChoice.EMAIL:
                    logger.info(f"Challenge via EMAIL requested for {insta_username}")
                    logger.info(f"📧 Checking email {email_address} for verification code...")
                    code = get_code_from_email_for_challenge(insta_username, email_address, email_password)
                    if code:
                        logger.info(f"✅ Challenge code retrieved for {insta_username}: {code}")
                        return code
                    else:
                        logger.warning(f"❌ Failed to retrieve challenge code for {insta_username}")
                        return False
                elif choice == ChallengeChoice.SMS:
                    logger.warning(f"⚠️ Challenge via SMS requested for {insta_username} (SMS verification not supported)")
                    return False
                else:
                    logger.warning(f"⚠️ Unknown challenge choice: {choice}")
                    return False
            
            cl.challenge_code_handler = challenge_code_handler
            logger.info(f"✅ Challenge code handler set for {username} using email {email_address}")
        
        # Add delay to avoid rate limiting
        await asyncio.sleep(random.uniform(1, 3))
        
        # Login with credentials - use asyncio.to_thread to avoid blocking
        # This ensures NO PROMPTS - credentials come from verify.csv
        try:
            await asyncio.to_thread(cl.login, account['username'], account['password'])
            logger.info(f"✅ Login successful for {username}")
            
            # Ensure sessions directory exists
            ensure_sessions_dir()
            
            # Save session (also make async)
            session_path = os.path.join(SESSIONS_DIR, f"{username}_session.json")
            await asyncio.to_thread(cl.dump_settings, session_path)
            logger.info(f"✅ Session saved for {username}")
            
            # Restore original input and getpass if we monkey-patched them
            if hasattr(cl, '_original_input'):
                import builtins
                import getpass
                builtins.input = cl._original_input
                if hasattr(cl, '_original_getpass'):
                    getpass.getpass = cl._original_getpass
                logger.debug(f"Restored original input() and getpass() for {username}")
            
            await asyncio.sleep(random.uniform(1, 2))
            return True
        except Exception as login_error:
            # Restore original input and getpass if we monkey-patched them
            if hasattr(cl, '_original_input'):
                import builtins
                import getpass
                builtins.input = cl._original_input
                if hasattr(cl, '_original_getpass'):
                    getpass.getpass = cl._original_getpass
                logger.debug(f"Restored original input() and getpass() for {username} after error")
            
            error_msg = str(login_error).lower()
            if "suspended" in error_msg or "challenge" in error_msg:
                logger.warning(f"⚠️ Login failed for {username}: {login_error}")
            else:
                logger.error(f"❌ Login error for {username}: {login_error}")
            return False
            
    except Exception as e:
        # Restore original input and getpass if we monkey-patched them
        try:
            if 'cl' in locals() and hasattr(cl, '_original_input'):
                import builtins
                import getpass
                builtins.input = cl._original_input
                if hasattr(cl, '_original_getpass'):
                    getpass.getpass = cl._original_getpass
                logger.debug(f"Restored original input() and getpass() for {username} after exception")
        except:
            pass
        
        logger.error(f"❌ Error during relogin for {username}: {e}")
        import traceback
        logger.error(traceback.format_exc())
        return False


def get_session_files() -> Dict[str, Dict]:
    """
    Get all session files from sessions directory.
    Returns dict mapping username to session info.
    """
    sessions = {}
    if not os.path.exists(SESSIONS_DIR):
        logger.warning(f"Sessions directory not found: {SESSIONS_DIR}")
        return sessions
    
    pattern = os.path.join(SESSIONS_DIR, "*_session.json")
    session_files = glob.glob(pattern)
    
    for session_file in session_files:
        try:
            filename = os.path.basename(session_file)
            username = filename.replace("_session.json", "")
            
            with open(session_file, 'r', encoding='utf-8') as f:
                session_data = json.load(f)
            
            user_id = session_data.get("authorization_data", {}).get("ds_user_id", "")
            
            sessions[username] = {
                "session_file": session_file,
                "username": username,
                "user_id": user_id,
                "account_id": user_id if user_id else username
            }
        except Exception as e:
            logger.error(f"Error reading session file {session_file}: {e}")
    
    logger.info(f"Loaded {len(sessions)} session files")
    return sessions


async def download_media(url: str, folder: Path) -> Path:
    """
    Download media from URL to temporary folder.
    Returns path to downloaded file.
    """
    try:
        async with httpx.AsyncClient(timeout=30.0, follow_redirects=True) as client:
            response = await client.get(url)
            response.raise_for_status()
            
            # Get file extension from URL or content-type
            ext = ".jpg"  # default
            content_type = response.headers.get("content-type", "").lower()
            if "png" in content_type or ".png" in url.lower():
                ext = ".png"
            elif "jpeg" in content_type or "jpg" in content_type or ".jpg" in url.lower() or ".jpeg" in url.lower():
                ext = ".jpg"
            elif "mp4" in content_type or ".mp4" in url.lower():
                ext = ".mp4"
            elif "webp" in content_type or ".webp" in url.lower():
                ext = ".webp"
            
            # Generate filename
            filename = f"{int(datetime.now().timestamp() * 1000)}_{random.randint(1000, 9999)}{ext}"
            filepath = folder / filename
            
            # Save file
            with open(filepath, 'wb') as f:
                f.write(response.content)
            
            logger.debug(f"Downloaded media: {url} -> {filepath}")
            return filepath
            
    except Exception as e:
        logger.error(f"Error downloading media from {url}: {e}")
        raise


def load_proxies_from_file() -> List[str]:
    """
    Load proxies from proxies.txt file.
    Each line should contain one proxy in format: scheme://user:pass@host:port
    Returns empty list if file doesn't exist or is empty.
    """
    global _proxies_cache
    
    # Return cached proxies if available
    if _proxies_cache is not None:
        return _proxies_cache
    
    proxies = []
    if os.path.exists(PROXIES_TXT):
        try:
            with open(PROXIES_TXT, 'r', encoding='utf-8') as f:
                for line in f:
                    line = line.strip()
                    # Skip empty lines and comments
                    if line and not line.startswith('#'):
                        proxies.append(line)
            logger.info(f"✅ Loaded {len(proxies)} proxies from {PROXIES_TXT}")
        except Exception as e:
            logger.error(f"Error loading proxies from {PROXIES_TXT}: {e}")
    else:
        logger.debug(f"Proxies file not found: {PROXIES_TXT}")
    
    # Cache the proxies
    _proxies_cache = proxies
    return proxies


def get_random_proxy() -> Optional[str]:
    """
    Get a random proxy from proxies.txt file.
    Returns None if no proxies are available.
    Ensures proxy has proper scheme (http:// or https://).
    """
    proxies = load_proxies_from_file()
    if proxies:
        proxy = random.choice(proxies)
        # Ensure proxy has scheme
        if proxy and not proxy.startswith(('http://', 'https://', 'socks5://', 'socks4://')):
            # Add http:// if no scheme
            proxy = f"http://{proxy}"
        logger.debug(f"Randomly selected proxy: {proxy[:50]}...")
        return proxy
    return None


async def load_client_from_session(session_file: str, username: str, retry_relogin: bool = True) -> Optional[Client]:
    """
    Load instagrapi Client from session file.
    First tries to use existing session by loading it, only relogins if session is invalid.
    Uses instagrapi for relogin (better challenge handling).
    """
    try:
        if not os.path.exists(session_file):
            logger.warning(f"Session file not found: {session_file}")
            if retry_relogin:
                if await relogin_account_with_instagrapi(username):
                    # Try again after relogin
                    return await load_client_from_session(session_file, username, retry_relogin=False)
            return None
        
        # Get random proxy if available
        proxy_url = get_random_proxy()
        
        # Initialize client (instagrapi is synchronous)
        try:
            cl = Client()
            
            # Set proxy BEFORE loading settings (instagrapi may need it during initialization)
            if proxy_url:
                try:
                    # Ensure proxy has proper scheme (instagrapi requires it)
                    if not proxy_url.startswith(('http://', 'https://', 'socks4://', 'socks5://')):
                        proxy_url = f"http://{proxy_url}"
                    
                    cl.set_proxy(proxy_url)
                    logger.info(f"🔥 Using proxy for {username}: {proxy_url[:50]}...")
                except Exception as proxy_error:
                    logger.warning(f"Failed to set proxy for {username}: {proxy_error}")
                    logger.warning(f"Proxy format was: {proxy_url[:50]}...")
            
            # Load session settings (after proxy is set)
            cl.load_settings(session_file)
            
            # Re-set proxy after loading settings (in case settings override it)
            if proxy_url:
                try:
                    # Ensure proxy has proper scheme (instagrapi requires it)
                    if not proxy_url.startswith(('http://', 'https://', 'socks4://', 'socks5://')):
                        proxy_url = f"http://{proxy_url}"
                    cl.set_proxy(proxy_url)
                except Exception as proxy_error:
                    logger.debug(f"Failed to re-set proxy after loading settings: {proxy_error}")
        except Exception as init_error:
            logger.error(f"Failed to initialize client for {username}: {init_error}")
            raise
        
        # First, try to check if we're already logged in (session might be valid)
        # Try to get account info without logging in first (instagrapi is synchronous, wrap with asyncio.to_thread)
        try:
            account_info = await asyncio.to_thread(cl.account_info)
            logger.info(f"✅ Session is valid for {username}, using existing session")
            return cl
        except (ClientLoginRequired, ChallengeRequired) as e:
            # Session is invalid, need to login
            logger.info(f"⚠️ Session invalid for {username}, need to login")
        except Exception as e:
            # Other error, might need login
            logger.debug(f"Could not verify session for {username}: {e}")
        
        # Get credentials from verify.csv (needed for login)
        account = get_account_from_verify_csv(username)
        if not account:
            logger.error(f"Cannot login {username}: account not found in verify.csv")
            if retry_relogin:
                if await relogin_account_with_instagrapi(username):
                    return await load_client_from_session(session_file, username, retry_relogin=False)
            return None
        
        # Try to login with credentials (only if session was invalid)
        # instagrapi login is synchronous, wrap with asyncio.to_thread
        try:
            await asyncio.to_thread(cl.login, account['username'], account['password'])
            logger.info(f"✅ Logged in {username} using credentials from verify.csv")
            
            # Verify login by getting account info
            try:
                account_info = await asyncio.to_thread(cl.account_info)
                logger.info(f"✅ Verified login for {username}")
                return cl
            except Exception as e:
                logger.warning(f"⚠️ Login verification failed for {username}: {e}")
                # Continue anyway - login might have succeeded
                return cl
                
        except (ClientLoginRequired, ChallengeRequired) as e:
            logger.warning(f"⚠️ Login failed for {username}: {e}")
            # Only relogin if login fails with challenge
            if retry_relogin:
                logger.info(f"🔄 Login failed, attempting relogin with instagrapi...")
                if await relogin_account_with_instagrapi(username):
                    # After relogin, try to load client again
                    return await load_client_from_session(session_file, username, retry_relogin=False)
                else:
                    # Relogin failed - mark as suspended
                    logger.error(f"🚫 Relogin failed for {username} - marking account as suspended")
                    mark_account_as_suspended(username, f"Login failed - relogin failed after {type(e).__name__}")
            return None
        except Exception as e:
            error_msg = str(e).lower()
            # Check if it's a login-related error
            if "login" in error_msg or "challenge" in error_msg or "session" in error_msg:
                logger.warning(f"⚠️ Login error for {username}: {e}")
                if retry_relogin:
                    logger.info(f"🔄 Attempting relogin with instagrapi...")
                    if await relogin_account_with_instagrapi(username):
                        return await load_client_from_session(session_file, username, retry_relogin=False)
                    else:
                        # Relogin failed - mark as suspended
                        logger.error(f"🚫 Relogin failed for {username} - marking account as suspended")
                        mark_account_as_suspended(username, f"Login error - relogin failed: {error_msg}")
            else:
                logger.error(f"Error logging in {username}: {e}")
            return None
            
    except Exception as e:
        logger.error(f"Error loading client from session {session_file}: {e}")
        import traceback
        logger.error(traceback.format_exc())
        return None


async def post_carousel_to_account(
    client: Client,
    username: str,
    media_paths: List[Path],
    caption: str,
    session_file: Optional[str] = None,
    attempt: int = 1,
    max_attempts: int = 3
) -> Dict:
    """
    Post carousel (album) to Instagram account with retry logic.
    """
    try:
        # Ensure caption is a string (not None or empty)
        caption_str = caption if caption else ""
        
        # Log caption being used
        if caption_str:
            logger.info(f"📝 Posting to {username} with caption: {caption_str[:100]}..." if len(caption_str) > 100 else f"📝 Posting to {username} with caption: {caption_str}")
        else:
            logger.info(f"📝 Posting to {username} without caption")
        
        # Upload album using instagrapi (synchronous, wrap with asyncio.to_thread)
        media = await asyncio.to_thread(
            client.album_upload,
            paths=media_paths,
            caption=caption_str
        )
        
        return {
            "username": username,
            "status": "success",
            "media_id": str(media.pk),
            "media_code": media.code,
            "message": "Carousel posted successfully",
            "attempt": attempt
        }
        
    except ClientLoginRequired:
        # Session expired - try to relogin using instagrapi and retry
        logger.warning(f"⚠️ Session expired for {username}, attempting relogin with instagrapi...")
        
        if session_file and attempt == 1:
            # Try to relogin using instagrapi
            if await relogin_account_with_instagrapi(username):
                logger.info(f"✅ Relogin successful for {username}, retrying post...")
                # Reload client from session after relogin
                new_client = await load_client_from_session(session_file, username, retry_relogin=False)
                if new_client:
                    # Retry with new client
                    await asyncio.sleep(random.uniform(2, 5))  # Small delay before retry
                    return await post_carousel_to_account(
                        new_client, 
                        username, 
                        media_paths, 
                        caption_str, 
                        session_file, 
                        attempt + 1, 
                        max_attempts
                    )
        
        # Relogin failed - mark account as suspended
        logger.error(f"🚫 Relogin failed for {username} - marking account as suspended")
        mark_account_as_suspended(username, "Session expired - relogin failed after LoginRequired")
        
        return {
            "username": username,
            "status": "error",
            "message": "Session expired - login required (relogin failed, account marked as suspended)",
            "attempt": attempt
        }
    except (PleaseWaitFewMinutes, RateLimitError) as e:
        wait_time = random.uniform(60, 120)  # Wait 1-2 minutes
        if attempt < max_attempts:
            logger.warning(f"Rate limited for {username}, waiting {wait_time:.1f}s before retry {attempt + 1}/{max_attempts}")
            await asyncio.sleep(wait_time)
            return await post_carousel_to_account(client, username, media_paths, caption, session_file, attempt + 1, max_attempts)
        return {
            "username": username,
            "status": "error",
            "message": f"Rate limited after {max_attempts} attempts: {str(e)}",
            "attempt": attempt
        }
    except ChallengeRequired as e:
        return {
            "username": username,
            "status": "error",
            "message": f"Challenge required: {str(e)}",
            "attempt": attempt
        }
    except Exception as e:
        error_msg = str(e).lower()
        if "rate limit" in error_msg or "too many" in error_msg:
            wait_time = random.uniform(30, 60)
            if attempt < max_attempts:
                logger.warning(f"Rate limit error for {username}, waiting {wait_time:.1f}s before retry")
                await asyncio.sleep(wait_time)
                return await post_carousel_to_account(client, username, media_paths, caption, session_file, attempt + 1, max_attempts)
        
        logger.error(f"Error posting carousel for {username} (attempt {attempt}): {e}")
        return {
            "username": username,
            "status": "error",
            "message": str(e),
            "attempt": attempt
        }


async def process_carousel_post(
    sessions_list: List[str],
    urls_list: List[str],
    caption: str = "",
    order_name: str = ""
) -> Dict:
    """
    Process carousel post - download media and post to all sessions.
    """
    # Generate unique ID and timestamps
    post_id = str(uuid.uuid4())
    created_at = datetime.now().isoformat()
    updated_at = created_at
    
    # Get all available sessions
    all_sessions = get_session_files()
    
    if not all_sessions:
        raise HTTPException(
            status_code=404,
            detail="No session files found in sessions directory"
        )
    
    # Parse and validate sessions
    selected_sessions = {}
    for session_name in sessions_list:
        session_name = session_name.strip()
        if session_name in all_sessions:
            selected_sessions[session_name] = all_sessions[session_name]
        else:
            logger.warning(f"Session {session_name} not found, skipping")
    
    if not selected_sessions:
        raise HTTPException(
            status_code=404,
            detail=f"No valid sessions found from: {sessions_list}"
        )
    
    logger.info(f"📸 Posting carousel to {len(selected_sessions)} accounts")
    
    # Create temporary directory that will stay alive until all posts complete
    temp_dir = tempfile.mkdtemp()
    temp_path = Path(temp_dir)
    
    try:
        # Download all media files
        media_paths = []
        failed_urls = []
        
        # Download media with progress tracking
        for i, url in enumerate(urls_list, 1):
            url = url.strip()
            if not url:
                continue
            try:
                logger.info(f"⬇️ Downloading media {i}/{len(urls_list)}: {url[:50]}...")
                media_path = await download_media(url, temp_path)
                media_paths.append(media_path)
                logger.info(f"✅ Successfully downloaded media {i}/{len(urls_list)}")
                # Small delay between downloads
                await asyncio.sleep(random.uniform(0.5, 1.5))
            except Exception as e:
                logger.warning(f"⚠️ Failed to download media {i}/{len(urls_list)} from {url}: {e}")
                failed_urls.append(url)
                # Continue with other URLs instead of failing completely
                continue
        
        if not media_paths:
            # Clean up temp directory before raising error
            import shutil
            shutil.rmtree(temp_dir, ignore_errors=True)
            raise HTTPException(
                status_code=400,
                detail=f"No media files downloaded successfully. Failed URLs: {failed_urls}"
            )
        
        if failed_urls:
            logger.warning(f"⚠️ {len(failed_urls)} URL(s) failed to download, but continuing with {len(media_paths)} successful download(s)")
            logger.warning(f"Failed URLs: {', '.join(failed_urls)}")
        
        logger.info(f"✅ Downloaded {len(media_paths)} media files")
        
        # Use provided caption (single caption for all posts)
        
        # Post to all selected accounts with advanced ban prevention
        results = []
        total_accounts = len(selected_sessions)
        
        for idx, (username, session_info) in enumerate(selected_sessions.items(), 1):
            try:
                logger.info(f"📤 [{idx}/{total_accounts}] Processing account: {username}")
                
                # Load client from session (with auto-relogin)
                client = await load_client_from_session(session_info["session_file"], username)
                if not client:
                    results.append({
                        "username": username,
                        "status": "error",
                        "message": "Failed to load client from session (relogin failed)"
                    })
                    continue
                
                # Random delay before posting (stagger posts)
                if idx > 1:
                    delay = random.uniform(3, 8)  # 3-8 seconds between posts
                    logger.debug(f"⏳ Waiting {delay:.1f}s before posting to {username}...")
                    await asyncio.sleep(delay)
                
                # Post carousel (pass session_file for relogin on LoginRequired)
                result = await post_carousel_to_account(
                    client,
                    username,
                    media_paths,
                    caption,
                    session_info["session_file"]  # Pass session file for relogin
                )
                results.append(result)
                
                # Log result
                if result.get("status") == "success":
                    logger.info(f"✅ Successfully posted to {username}")
                else:
                    logger.warning(f"❌ Failed to post to {username}: {result.get('message')}")
                
                # Additional delay after posting (vary by account)
                post_delay = random.uniform(2, 5)
                await asyncio.sleep(post_delay)
                
            except Exception as e:
                logger.error(f"Error processing account {username}: {e}")
                results.append({
                    "username": username,
                    "status": "error",
                    "message": str(e)
                })
        
        # Calculate summary
        success_count = len([r for r in results if r.get("status") == "success"])
        failed_count = len([r for r in results if r.get("status") == "error"])
        completed_at = datetime.now().isoformat()
        
        status = "completed" if success_count > 0 else "failed"
        
        return {
            "id": post_id,
            "status": status,
            "created_at": created_at,
            "updated_at": completed_at,
            "completed_at": completed_at,
            "results": results,
            "summary": {
                "total": len(results),
                "success": success_count,
                "failed": failed_count,
                "order_name": order_name
            }
        }
    finally:
        # Clean up temporary directory after all posts are complete
        import shutil
        try:
            shutil.rmtree(temp_dir, ignore_errors=True)
            logger.debug(f"Cleaned up temporary directory: {temp_dir}")
        except Exception as e:
            logger.warning(f"Failed to clean up temporary directory {temp_dir}: {e}")


@media_router.post(
    "/carousel",
    response_model=Union[CarouselPostResponse, BackgroundJobResponse],
    tags=["Media"],
)
async def post_carousel(
    request: SimpleCarouselRequest = Body(...)
):
    """
    Post carousel (album) to multiple Instagram accounts.
    
    **Simplified input:**
    - **sessions**: Comma-separated session names (e.g., "session1,session2,session3")
    - **urls**: Comma-separated image URLs (e.g., "url1,url2,url3")
    - **caption** or **captions**: Single caption for all posts (optional, accepts both field names)
    - **order_name**: Optional order name for tracking
    
    **Features:**
    - Automatic status tracking (incomplete -> processing -> completed)
    - Auto-relogin if session is inactive (uses verify.csv)
    - Advanced ban prevention (random delays, retry logic)
    - Async processing for better performance
    """
    try:
        # Parse comma-separated inputs
        sessions_list = [s.strip() for s in request.sessions.split(",") if s.strip()]
        urls_list = [u.strip() for u in request.urls.split(",") if u.strip()]
        # Get caption from either 'caption' or 'captions' field (handles both singular and plural)
        caption = request.get_caption()
        
        if not sessions_list:
            raise HTTPException(status_code=400, detail="No sessions provided")
        if not urls_list:
            raise HTTPException(status_code=400, detail="No URLs provided")
        
        logger.info(f"🚀 Starting carousel post: {len(sessions_list)} sessions, {len(urls_list)} media URLs")
        if caption:
            logger.info(f"📝 Caption: {caption[:100]}..." if len(caption) > 100 else f"📝 Caption: {caption}")
        else:
            logger.info("📝 No caption provided (will post without caption)")
        
        # Always run in background (default behavior - not exposed in API)
        job = await background_tasks.create_task(
            name="media:carousel",
            coro_func=process_carousel_post,
            sessions_list=sessions_list,
            urls_list=urls_list,
            caption=caption,
            order_name=request.order_name or ""
        )
        return {
            "job_id": job["id"],
            "status": job["status"],
            "message": "Carousel job queued",
            "total_sessions": len(sessions_list),
            "total_media": len(urls_list),
        }
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error in post_carousel endpoint: {e}")
        import traceback
        logger.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail=f"Failed to post carousel: {str(e)}")


@media_router.get("/sessions", tags=["Media"])
async def get_available_sessions():
    """
    Get list of all available sessions (accounts) that can be used for posting.
    Returns session names that can be used in the 'sessions' parameter.
    """
    sessions = get_session_files()
    
    session_list = []
    for username, session_info in sessions.items():
        session_list.append({
            "session_name": username,  # Use this in 'sessions' parameter
            "username": session_info["username"],
            "user_id": session_info["user_id"],
            "account_id": session_info["account_id"]
        })
    
    return {
        "status": "ok",
        "sessions": session_list,
        "total": len(session_list),
        "usage": "Use 'session_name' values in the 'sessions' parameter (comma-separated)"
    }
